"""
Get set

Author: Maddy Guthridge
Date:
* 2023-08-18 = initial version
* 2025-01-26 = improvements to checks

A small program to check basic setup of students' working environments

###### IMPORTANT ######
There's no need for you to understand this code.
"""

import os
import shutil
import sys
from pathlib import Path


def check_python():
    """
    Check for reasonable Python version
    """
    print("Checking Python...")
    if sys.version_info < (3, 13):  # noqa: UP036
        print("It looks like you're using an old version of Python!")
        print(
            f"Expected version ^3.13, got version "
            f"{sys.version_info.major}.{sys.version_info.minor}"
        )
        print(VERSION_FIXES)
        exit(1)


def check_path():
    """
    Check that executables are accessible from the command line
    """
    print("Checking path...")
    if shutil.which("uvx") is None:
        # uv not on Path
        print("We couldn't find some tools you need on the command line")
        print("(failed to find `uvx` command)")
        if (
            sys.platform == "darwin"  # MacOS
            or sys.platform == "win32"  # Windows
        ):
            print(ASK_FOR_HELP)
        elif sys.platform == "linux":
            print(" -> Make sure `$HOME/.local/bin` is in your $PATH variable")
            print(" -> Restart your terminal")
        else:
            print(f" -> Unrecognised platform ({sys.platform}).")
            print(" -> Ask your tutor for help.")
        exit(1)


def check_dir():
    """
    Check that the program was run from the right directory
    """
    print("Checking working directory...")
    try:
        open("get_set.py").close()
    except FileNotFoundError:
        print("It looks like you haven't opened the correct folder in VS Code")
        print(DIR_FIXES)
        exit(1)


def configure_code():
    """
    Configure VS Code by copying this project's settings into the global
    settings.
    """
    print("Configuring VS Code...")
    global_settings_file = VSCODE_SETTINGS[sys.platform]
    local_settings_file = ".vscode/settings.json"
    with open(local_settings_file) as f:
        local_settings = f.read()

    try:
        with open(global_settings_file) as f:
            existing_global_settings = f.read()
        if existing_global_settings == local_settings.strip():
            print("VS Code already configured.")
            return
        if existing_global_settings.strip() not in [
            "",
            "{}",
        ]:
            print("You already have some VS Code settings configured!")
            print("Do you want to overwrite them with the COMP1010 defaults?")
            answer = input("[y/n]: ").strip().lower()
            if answer not in ["y", "yes"]:
                print("Your VS Code settings were not modified.")
                return
            else:
                print("Creating a backup of your old settings...")
                backup = global_settings_file.parent / "settings.backup.json"
                with open(backup, "w") as f:
                    f.write(existing_global_settings)
                print(f"Backup created in: {backup}")
    except FileNotFoundError:
        pass
    with open(global_settings_file, "w") as f:
        f.write(local_settings)
    print("VS Code configured successfully!")


def main():
    """
    Entrypoint to the program
    """
    check_python()
    check_path()
    check_dir()
    configure_code()
    print(SUCCESS)


VSCODE_SETTINGS = {
    "win32": Path(f"{os.getenv('APPDATA')}/Code/User/settings.json"),
    "darwin": Path(
        f"{os.getenv('HOME')}/Library/Application Support/Code/User/settings.json"
    ),  # noqa: E501
    "linux": Path(f"{os.getenv('HOME')}/.config/Code/User/settings.json"),
}


ASK_FOR_HELP = """
 -> If you're stuck or confused, your tutor will be happy to help!
 -> Not during tutorial time? Post on the forum! There's a link in the README.
"""


VERSION_FIXES = (
    """
 -> Did you make sure to install Python?
 -> Did VS Code choose the right Python version (see bottom right)?
 -> If you're on Windows, try rebooting your system."""
    + ASK_FOR_HELP
)

DIR_FIXES = (
    """
 -> Make sure you follow the instructions in the README."""
    + ASK_FOR_HELP
)


SUCCESS = r"""
     _________________________
   < Congrats! You're all set! >
     -------------------------
            \   ^__^
             \  (oo)\_______
                (__)\       )\/\
                    ||----- |
                    ||     ||

"""


if __name__ == "__main__":
    main()
